/*____________________________________________________________________________
	Copyright (C) 2000 Networks Associates Technology, Inc.
	All rights reserved.

	pgpCache.c - indexed data cache used for passkeys
	

	$Id: pgpCache.c,v 1.2 2001/03/12 21:23:23 pbj Exp $
____________________________________________________________________________*/

#include "pgpUDx.h"
#include "PGPsdkDriver.h"
#include "pgpCache.h"
#include "pgpMisc.h"


//	______________________________________________________
//
//	decrement the second timer and purge timed-out entries

static VOID
sCacheTick (
	IN PPGPCACHE	ppc)
{
	PPGPCACHEITEM*	pppci		= &(ppc->pitemFirst);
	PPGPCACHEITEM	ppciThis	= NULL;
	PPGPCACHEITEM	ppciTemp	= NULL;

	while (*pppci)
	{
		ppciThis = *pppci;

		// decrement the counter
		--(ppciThis->lSecondsRemaining);

		// if timed-out, zero item.
		// it will be removed on next call to cleanup function
		if (ppciThis->lSecondsRemaining <= 0)
		{
			ppciTemp = ppciThis->pitemNext;
			memset (ppciThis, 0, sizeof(PGPCACHEITEM));
			ppciThis->pitemNext = ppciTemp;
		}
		// otherwise just step to next entry in list
		else
			pppci = &(ppciThis->pitemNext);
	}
}


//	______________________________________________________
//
//	remove any unused items from the list

static VOID
sCleanupCache (
	IN PPGPCACHE	ppc)
{
	PPGPCACHEITEM*	pppci		= &(ppc->pitemFirst);
	PPGPCACHEITEM	ppciThis	= NULL;

	while (*pppci)
	{
		ppciThis = *pppci;

		// if zero, this item is not used
		if (ppciThis->lSecondsRemaining <= 0)
		{
			*pppci = ppciThis->pitemNext;
			memset (ppciThis, 0, sizeof(PGPCACHEITEM));
			pgpDriverSecureFree (ppciThis);
		}
		// otherwise just step to next entry in list
		else
			pppci = &(ppciThis->pitemNext);
	}
}


//	______________________________________________________
//
//	purge the entire cache

static VOID
sPurgeCache (
	IN PPGPCACHE	ppc)
{
	PPGPCACHEITEM	ppci		= ppc->pitemFirst;
	PPGPCACHEITEM	ppciNext	= NULL;

	while (ppci)
	{
		ppciNext = ppci->pitemNext;
		memset (ppci, 0, sizeof(PGPCACHEITEM));
		pgpDriverSecureFree (ppci);

		ppci = ppciNext;
	}

	ppc->pitemFirst = NULL;
}


//	______________________________________________________
//
//	find the indexed value in the cache

static VOID
sQueryCache (
	IN PPGPCACHE		ppc,
	IN PPGPCACHESTRUCT	ppcs)
{
	PPGPCACHEITEM	ppci		= ppc->pitemFirst;

	while (ppci)
	{
		if (ppci->ulIndexLength == ppcs->ulIndexLength)
		{
			if (memcmp (ppci->ucIndex, 
					ppcs->ucIndex, ppci->ulIndexLength) == 0)
			{
				// copy cache contents for passing back
				ppcs->ulDataLength = ppci->ulDataLength;
				memcpy (ppcs->ucData, ppci->ucData, ppci->ulDataLength);
				ppcs->ulError = kPGPUDError_NoErr;

				// reset timeout counter
				ppci->lSecondsRemaining = (LONG)(ppcs->ulSeconds);
				return;
			}
		}

		ppci = ppci->pitemNext;
	}

	ppcs->ulError = kPGPUDError_ItemNotFound;
}


//	______________________________________________________
//
//	find the indexed value in the cache

static VOID
sSetCacheValue (
	IN PPGPCACHE		ppc,
	IN PPGPCACHESTRUCT	ppcs)
{
	PPGPCACHEITEM	ppci		= ppc->pitemFirst;
	PPGPCACHEITEM	ppciNew		= NULL;

	// find if index is already in list
	while (ppci)
	{
		if (ppci->ulIndexLength == ppcs->ulIndexLength)
		{
			if (memcmp (ppci->ucIndex, 
					ppcs->ucIndex, ppci->ulIndexLength) == 0)
			{
				// replace cache contents
				ppci->ulDataLength = ppcs->ulDataLength;
				memcpy (ppci->ucData, ppcs->ucData, ppcs->ulDataLength);
				ppcs->ulError = kPGPUDError_NoErr;

				// reset timeout counter
				ppci->lSecondsRemaining = (LONG)(ppcs->ulSeconds);
				return;
			}
		}

		ppci = ppci->pitemNext;
	}

	// otherwise, insert item into list
	ppciNew = pgpDriverSecureAlloc (sizeof(PGPCACHEITEM));
	if (ppciNew)
	{
		// fill in data
		ppciNew->ulIndexLength = ppcs->ulIndexLength;
		memcpy (ppciNew->ucIndex, ppcs->ucIndex, ppcs->ulIndexLength);
		ppciNew->ulDataLength = ppcs->ulDataLength;
		memcpy (ppciNew->ucData, ppcs->ucData, ppcs->ulDataLength);

		// set timeout
		ppciNew->lSecondsRemaining = (LONG)(ppcs->ulSeconds);

		// add to front of list
		ppciNew->pitemNext = ppc->pitemFirst;
		ppc->pitemFirst = ppciNew;

		ppcs->ulError = kPGPUDError_NoErr;
	}
	else
		ppcs->ulError = kPGPUDError_MemAllocError;
}


//	______________________________________________________
//
//	process a timer event -- purge expired cache entries

VOID
pgpProcessCacheTimerEvent (
	IN PPGPCACHE		ppc,
	IN PVOID			pCriticalSection)
{
	pgpDriverEnterCriticalSection (pCriticalSection, SPINLOCK);
	sCacheTick (ppc);
	pgpDriverLeaveCriticalSection (pCriticalSection, SPINLOCK);
}


//	______________________________________________________
//
//	Initialize the passkey cache

VOID
pgpInitCache (
	IN PPGPCACHE		ppc)
{
	ppc->pitemFirst = NULL;
}


//	______________________________________________________
//
//	This is the function that processes the DeviceIoControl
//	call which is coming from an application.

VOID
pgpCacheProcessOperation (
	IN PPGPCACHE		ppc,
	IN PPGPCACHESTRUCT	ppcs,
	IN PVOID			pCriticalSection)
{
	switch (ppcs->ulOperation) {
	case kPGPUDOperation_SetCacheValue :
		pgpDriverEnterCriticalSection (pCriticalSection, MUTEX);
		sCleanupCache (ppc);
		sSetCacheValue (ppc, ppcs);
		pgpDriverLeaveCriticalSection (pCriticalSection, MUTEX);
		break;

	case kPGPUDOperation_QueryCacheValue :
		pgpDriverEnterCriticalSection (pCriticalSection, MUTEX);
		sCleanupCache (ppc);
		sQueryCache (ppc, ppcs);
		pgpDriverLeaveCriticalSection (pCriticalSection, MUTEX);
		break;

	case kPGPUDOperation_PurgeCache :
		pgpDriverEnterCriticalSection (pCriticalSection, MUTEX);
		sPurgeCache (ppc);
		ppcs->ulError = kPGPUDError_NoErr;
		pgpDriverLeaveCriticalSection (pCriticalSection, MUTEX);
		break;

	default :
		ppcs->ulError = kPGPUDError_UndefinedOperation;
		break;
	}
}

